% Example4_7.m  Nonlinear Tikhonov regularization

q0 = 10;       % W/m^2 - used for initial guess

itmax = 20;        % iterations for iterative IHCP soln
cnvrg_tol = 1e-2;  % stop when norm(dq)/max(q) < cnvrg_tol

L_d = 0.050;       % m - thickness of the plate
nseg = 20;         % number of segments for the finite volume solution
dx = L_d/nseg;

t = [ -34.0 -25.5 -17.0 -8.5    0.0   8.5  17.0  25.5  34.0  42.5  51.0 ... 
       59.5  68.0  76.5  85.0  93.5 102.0 110.5 119.0 127.5 136.0 144.5 ... 
      153.0 161.5 170.0 178.5 187.0 195.5 204.0 212.5 221.0 229.5 238.0 ...
      246.5 255.0 263.5 272.0 280.5 289.0 297.5 306.0 314.5 323.0 331.5 ]';
TL = [ 25.00   25.00  25.00  25.00 25.00 25.00 25.03 25.12 25.37 25.92 ... 
       26.89   28.32  30.22  32.57 35.35 38.54 42.12 46.05 50.23 54.53 ... 
       58.75   62.77  66.50  69.95 73.19 76.26 79.19 81.98 84.61 87.05 ...
       89.31   91.35  93.17  94.78 96.17 97.37 98.39 99.25 99.98 100.58 ...
       101.04 101.37 101.59 101.72 ]';   % C
qbc = [ 0.0  0.0  0.0  0.0  0.0  2.0  7.0 14.8 28.2 39.0 49.1 58.6 67.4 ...
       76.2 85.8 94.2 99.3 98.8 92.0 82.3 73.4 68.7 66.2 64.2 62.1 59.2 ...
       55.1 50.0 44.7 39.6 34.5 29.2 24.2 20.3 17.5 15.4 13.4 10.8  5.9 ...
        1.9  1.3  0.4  0.0  0.0 ]'*1000;   % W/m^2

n = length(TL)-1;  % first one is inital condition
                   % n is number of heat flux components 

sigma = 0.25;   % in degrees C - this is the stdev of measurement noise
rng(129);      % random number seed - always gets the same noise
noise = randn( n, 1);  % for noising all the data except I.C.

q_gaps = ( qbc(1:end-1)+qbc(2:end))/2;  % these are average fluxes over each time interval
T_test = T_finitevolumeimplicit( t, TL, q_gaps, L_d, nseg );
figure(2); hold on; plot( t, TL, 'b-', t(2:end), T_test,'bv' ); hold off; figure(2);

T0 = TL(1);   % use as the initial condition
T = zeros(n+1,1);
T(1) = T0;  % only need the initial condition....

Yvec = TL(2:end) + noise*sigma;  % add the measurement noise

q = ones(n,1)*q0/2;  % initial "guess" for the heat flux vector q(t)
Tvec = T_finitevolumeimplicit( t, T, q, L_d, nseg );
resid = (Yvec - Tvec);  
S0 = resid' * resid;

H = eye( n, n );     % for zeroth order Tikhonov regularization
if( sigma == 0 )
    alpha_Tik = 0.1e-10; % Tikhonov regularization coefficient
else
    alpha_Tik =11e-10; % Tikhonov regularization coefficient
end


% iterations for inverse solution
time_total = 0;      % monitor execution time
for iter = 1:itmax
    tic    % time the iteration
    [ X, Tvec ] = computeX( t, T, [q], L_d, n );
    resid = (Yvec - Tvec);  
    S(iter) = resid' * resid;
  
    lamda = (resid' * X * X' * resid) / S(iter);
    if(iter == 1)
        lamda0 = lamda;  % copy the first value
    end
    
        lamda = alpha_Tik;
        dq = (X'*X + H'*H*lamda) \ ((X' * resid) - alpha_Tik*H'*H*q);
        
        q = q + dq;
    
%     % re-compute S(iter) with the final q + dq
    Tvec = T_finitevolumeimplicit( t, T, q, L_d, nseg );
    resid = (Yvec - Tvec);  
    S(iter) = resid' * resid;

    dtime = toc;
    time_total = time_total + dtime;
    condXTX = cond(X'*X+H'*H*lamda);
    metric = norm(dq)/max(q);
    
    fprintf(' %3d   %8.2e  %8.2e %6.3f  %6.3f %8.3e  %8.3e \n', ...
      iter, alpha_Tik, condXTX, S(iter), sqrt(S(iter)/length(Yvec)), metric, dtime )
    figure(3); semilogy([0:iter], [ S0 S(1:iter)], 'linewidth', 1.0 ); 
    xlabel('iteration'); ylabel('S'); figure(3);
    metric = norm(dq)/max(q);
    if( metric < cnvrg_tol )
        break;
    end
    T = [ T0; Tvec ];   % update though not needed
    
end
figure(90); clf; plot( t(2:end)-(t(2)-t(1))/2, q/1000, 'ro', ...
                       t, qbc/1000, 'b-', 'linewidth', 1.0 )
xlabel('time, s'); ylabel('Heat Flux, kW/m^2');
legend('Estimated', 'Exact')
figure(91); clf; plot( t(2:end), Tvec, 'ro', t(2:end), Yvec, 'b+', 'linewidth', 1.0 )
xlabel('time, s'); ylabel('Temperature, C');
legend('Estimated T', 'Yvec data', 'location', 'northwest')

fprintf('Iterations: %3d, Total time: %6.3f seconds, Final norm(dq)/max(q): %8.3e\n', ...
           iter,time_total, norm(dq)/max(q) );
       
fig =figure(90); 
text(270,75,sprintf('\\sigma_Y = %4.2f C',sigma) )
if( sigma== 0 )
    fname='Figure4_7';
else
    fname='Figure4_9';
end
print(fig,fullfile(pwd, fname),'-dtiff','-r600')

fig =figure(91); 
text(-10,90,sprintf('\\sigma_Y = %4.2f C',sigma) )
if( sigma== 0 )
    fname='Figure4_8';
else
    fname='Figure4_10';
end
print(fig,fullfile(pwd, fname),'-dtiff','-r600')

fig = figure(3); % this is the convergence plot
fname='Figure4_11';
xticks([0:iter])
     print(fig,fullfile(pwd, fname),'-dtiff','-r600')



function [ X, Tvec ] = computeX( t, T, q, L_d, nseg )
% use perturbation of each value in 'q' to get sensitivity matrix
dq = 500;   % absolute dq
pct = 0.01;  % percentage

dt = t(2) - t(1);
T0 = T(1);
n = length(T);  % first one is initial condition
X = zeros(n-1,n-1);
%
% first compute Tvec using the given q values
Tvec = T_finitevolumeimplicit( t, T, q, L_d, nseg );
T = Tvec;
for i = 1:n-1
    qpert = q;
    if( abs(q(i)) < dq )
        qpert(i) = qpert(i)+dq;
    else
        qpert(i) = qpert(i)*(1+pct);
    end
    
    Tnew = T_finitevolumeimplicit( t, T, qpert, L_d, nseg );
    
    if( abs(q(i)) < dq )
       Xcol = (Tnew-T)/dq;
    else
        Xcol = (Tnew-T)/q(i)/pct;
    end
    X(:,i) = Xcol;
end
end


function [ Tnew ] = T_finitevolumeimplicit( t, T, q, L_d, nseg )
% properties of Armco iron Vs. Temperature ( 0 < T < 125 C)
k_armco = @(T) 74.74-0.07693*T+4.5E-5*T.^2;
C_armco = @(T)(3.4+4.596E-3*T-7.87E-6*T.^2)*1E6;

% t,T are vector of length(n) 
% q is vector of length n-1
Tj = ones(nseg+1,1)*T(1);   % use the T(1) as initial condition
n = length(t);  % this is the number of times
dx = L_d / nseg;
dx2 = dx^2;
dt_data = t(2) - t(1);
Tnew = zeros( n-1, 1 );  % compute all except initial condition

nint = 5;  % number of subintervals for integration
dt = dt_data/nint;
tint = t(1); % sync with time for interpolations
for it = 1:n-1   % loop through all the times
    for in = 1:nint
        tint = tint+dt;
        q_int = pchip( [ t-dt_data/2], [0;q], tint );
        
        A = zeros( nseg+1, nseg +1);
        rhs = zeros( nseg + 1, 1);
        % compute properties based on current Told temps
        khalf = k_armco( (Tj(1:nseg)+Tj(2:nseg+1))/2 );
        C = C_armco( Tj );
        % fill the core of the A matrix
        for ir = 2:nseg
            A(ir,ir) = -(khalf(ir-1)/dx2 + khalf(ir)/dx2 + C(ir)/dt );
            A(ir,ir-1) = khalf(ir-1)/dx2;
            A(ir,ir+1) = khalf(ir)/dx2;
            rhs(ir) = -C(ir)/dt*Tj(ir);
        end
        % first row
        A(1,1) = -(2*khalf(1)/dx2 + C(1)/dt);
        A(1,2) =  2*khalf(1)/dx2;
 
 %       q_int = q(it);      % this is the average heat flux over tj < t < tjp1
        rhs(1) = -C(1)/dt*Tj(1) - 2*q_int/dx; 
        % last row
        A(nseg+1,nseg+1) = -(2*khalf(nseg)/dx2 + C(nseg+1)/dt);
        A(nseg+1,nseg) = (2*khalf(nseg)/dx2);
        rhs(nseg+1) = -C(nseg+1)/dt*Tj(nseg+1);
        % solve for one time step
        Tjp1 = A \ rhs;
        Tj = Tjp1;           % update for next time step
    end
    % extract results
    Tnew( it ) = Tjp1( nseg+1);
    % copy new into old for next time step
 
end
end
